%% Step 1: Determine if you have single-point or multi-point emissions
clear all;
close all;

% Specify the parent directory containing the CSV files
parentPath = '\\csunas01.colostate.edu\research\METEC\User\mbua\ADED\OTM33_Gaussian\Event Tables';

% Specify the folders to process
folders = {'MGGA'};

% Loop through each folder
for f = 1:numel(folders)
    folderName = folders{f};
    directory = fullfile(parentPath, folderName);
    
    % List all CSV files in the directory
    fileList = dir(fullfile(directory, '*.csv'));

    % Create a new directory to store split tables based on emission points
    outputDirectory = fullfile(parentPath, folderName, 'Emission Sources');
    if ~exist(outputDirectory, 'dir')
        mkdir(outputDirectory);
    end

% Loop through each file in the folder
for fileIdx = 1:numel(fileList)
    % Load the CSV file
    filePath = fullfile(directory, fileList(fileIdx).name);
    data = readtable(filePath);

    % Count the number of unique values in the 'EmissionPoint' column
    uniquePoints = unique(data.EmissionPoint);
    numUniquePoints = numel(uniquePoints);
    
    % Add a column containing the count of unique points
    data.CountUniquePoints = repmat(numUniquePoints, height(data), 1);

    % Create a new directory based on the number of emission points
    if numUniquePoints == 1
        outputSubDirectory = fullfile(outputDirectory, 'Single_Point');
    else
        outputSubDirectory = fullfile(outputDirectory, 'Multi_Point');
    end

    % Create the subdirectory if it doesn't exist
    if ~exist(outputSubDirectory, 'dir')
        mkdir(outputSubDirectory);
    end

    % Copy the file to the appropriate directory
    destinationFilePath = fullfile(outputSubDirectory, fileList(fileIdx).name);
    copyfile(filePath, destinationFilePath);
    
    % Write the updated data table to a new CSV file with the count column
    outputFileName = fullfile(outputSubDirectory, [fileList(fileIdx).name]);
    writetable(data, outputFileName);
end

    % Count the number of unique values in the 'EmissionPoint' column
    uniquePoints = unique(data.EmissionPoint);
    numUniquePoints = numel(uniquePoints);
    
    % Add a column containing the count of unique points
    data.CountUniquePoints = repmat(numUniquePoints, height(data), 1);

    % Create a new directory based on the number of emission points
    if numUniquePoints == 1
        outputSubDirectory = fullfile(outputDirectory, 'Single_Point');
    else
        outputSubDirectory = fullfile(outputDirectory, 'Multi_Point');
    end

    % Create the subdirectory if it doesn't exist
    if ~exist(outputSubDirectory, 'dir')
        mkdir(outputSubDirectory);
    end

    % Copy the file to the appropriate directory
    destinationFilePath = fullfile(outputSubDirectory, fileList(fileIdx).name);
    copyfile(filePath, destinationFilePath);
    
    % Write the updated data table to a new CSV file with the count column
    outputFileName = fullfile(outputSubDirectory, [fileList(fileIdx).name]);
    writetable(data, outputFileName);
end

%% Focus on Single_Point Sources first

% Step 2: Split into 5, 15 & 30 minute tables
clear all;
close all;

% Specify the parent directory containing the CSV files
parentPath = '\\csunas01.colostate.edu\research\METEC\User\mbua\ADED\OTM33_Gaussian\Event Tables';

% Specify the folders to process
folders = {'MGGA'};

% Loop through each folder
for f = 1:numel(folders)
    folderName = folders{f};
    path = fullfile(parentPath, folderName,'Emission Sources', 'Single_Point');
    
    % Step 1: For every event, determine 30-minute mini-tables
    % Initialize cell array to store mini-tables from all files
    allMiniTables = {};

    % Get a list of all files in the directory
    fileList = dir(fullfile(path, '*.csv'));

% Loop through each CSV file
for i = 1:numel(fileList)
    filename = fullfile(path, fileList(i).name);

    % Read the CSV file
    data = readtable(filename);
    % Remove the prefix from column names
    prefix = 'mean_'; % Specify the prefix to remove
    data.Properties.VariableNames = erase(data.Properties.VariableNames, prefix);

    % Determine which column to use based on the folder name
    if strcmp(folderName, 'Aeris_225') || strcmp(folderName, 'Aeris_259')
        % Use 'mean_CH4_ppm_' column
        ppm_column = 'CH4_ppm_';
    elseif strcmp(folderName, 'MGGA')
        % Use 'mean_x_CH4__ppm' column
        ppm_column = 'x_CH4__ppm';
    else
        error('Folder name not recognized.');
    end

    % Filter out data below 1900 ppb - instrument starting
    %data = data(data.(ppm_column) > 1.9, :);

    % Convert the 'Time' column to datetime format
    data.Time = datetime(data.Time);

    % Define time intervals (30 minutes) % change this to 5 or 15 minutes
    interval = minutes(30);

    % Compute the start and end times for each interval
    startTime = min(data.Time);
    endTime = max(data.Time);
    timeIntervals = startTime:interval:endTime;

    % Split the data into 15-minute mini-tables based on the 'Time' column
    miniTables = cell(1, numel(timeIntervals) - 1);

    for j = 1:numel(timeIntervals)-1
        startTimeInterval = timeIntervals(j);
        endTimeInterval = timeIntervals(j+1);

        % Filter the data for the current interval
        averaged_table = data(data.Time >= startTimeInterval & data.Time < endTimeInterval, :);

        averaged_table.Actual_Emission_kg_h = 0.001 * 60 * 16.04 * 1 * averaged_table.C1FlowAvg_slpm / (0.082057 * 293.15);

        % Store the mini-table in the cell array if it's 30 minutes long
        if height(averaged_table) >= 1800 %1800 for 30 minutes,  At least 900 rows are needed for a 15-minute interval- 1 hz frequency * time to determine minimum rows
            miniTables{j} = averaged_table;
        else
            fprintf('Skipping mini-table %d: Less than 15 minutes long.\n', j);
        end
    end

    % Add mini-tables from the current file to the cell array storing all mini-tables
    allMiniTables = [allMiniTables, miniTables];
end


    % Create a new folder named 5, 15 or 30 minutes" for each folder
    newFolderPath = fullfile(path, '30 minutes');
    if ~exist(newFolderPath, 'dir')
        mkdir(newFolderPath);
    end

    % Loop through each mini-table and calculate average Actual_Emission_kg_h
    for k = 1:numel(allMiniTables)
        miniTable = allMiniTables{k};

        % Check if the mini-table is valid (not empty)
        if ~isempty(miniTable)
            % Find unique rows based on the first column (EmissionPoint)
            [uniqueIndices, ~, uniqueValues] = unique(miniTable.EmissionPoint);

            % Calculate the sum of Actual_Emission_kg_h for each unique EmissionPoint
            Emissions = accumarray(uniqueValues, miniTable.Actual_Emission_kg_h, [], @(x) x(1));

            % Calculate the total sum of emissions across all unique emission points
            TotalEmissions = sum(Emissions);

            % Create a new column named 'TotalEmissions' with the total emissions for each row
            miniTable.TotalEmissions = repmat(TotalEmissions, size(miniTable, 1), 1);

            % Get the start and end times
            startTime = miniTable.Time(1);
            endTime = miniTable.Time(end);

            % Calculate the duration of the mini-table
            duration = endTime - startTime;

            % Check if the duration is approximately 30 minutes
            if duration >= minutes(29.5) && duration <= minutes(30.5) % change for 5 or 15 minutes
                % Create file names using start and end times
                startTimeStr = datestr(startTime, 'yyyy-mm-dd HH-MM-SS');
                endTimeStr = datestr(endTime, 'yyyy-mm-dd HH-MM-SS');
                fileName = sprintf('%s to %s.csv', startTimeStr, endTimeStr);
                filePath = fullfile(newFolderPath, fileName);

                % Write the mini-table to a CSV file
                writetable(miniTable, filePath);
            else
                fprintf('Skipping mini-table %d: Duration is not 30 minutes.\n', k);
            end
        end
    end
end

%% %% Step 3: Calculate bearing from data.Lat, data.Lon to data.Aeris/MGGA_lat_N, data.Aeris/MGGA_long_W
clear all;
close all;

% Define the function to calculate initial bearing
calculate_initial_bearing = @(lat1, lon1, lat2, lon2) mod(rad2deg(atan2(...
    sin(deg2rad(lon2 - lon1)) .* cos(deg2rad(lat2)), ...
    cos(deg2rad(lat1)) .* sin(deg2rad(lat2)) - sin(deg2rad(lat1)) .* cos(deg2rad(lat2)) .* cos(deg2rad(lon2 - lon1)))) ...
    + 360, 360);

% Specify the parent directory containing the CSV files
parentPath = '\\csunas01.colostate.edu\research\METEC\User\mbua\ADED\OTM33_Gaussian\Event Tables';

% Specify the main folders to process
folders = {'MGGA'};

% Specify the subfolders (time intervals)
time_intervals = { '30 minutes'};

% Loop through each folder
for f = 1:numel(folders)
    folderName = folders{f};

    % Loop through each time interval subfolder
    for t = 1:numel(time_intervals)
        timeFolder = time_intervals{t};
        
        % Construct the full path to the time interval folder
        path = fullfile(parentPath, folderName, 'Emission Sources', 'Single_Point', timeFolder);
        
        % Check if the folder exists before proceeding
        if ~isfolder(path)
            fprintf('Skipping: %s (Folder not found)\n', path);
            continue;
        end

        % Get a list of all CSV files in the directory
        fileList = dir(fullfile(path, '*.csv'));

        % Loop through each CSV file
        for i = 1:numel(fileList)
            filename = fullfile(path, fileList(i).name);

            % Read the CSV file
            data = readtable(filename);

            % Ensure required columns exist in the table
            if any(strcmp(data.Properties.VariableNames, 'Lat')) && any(strcmp(data.Properties.VariableNames, 'Lon'))
                
                % Specify column names based on the folder name
                if strcmp(folderName, 'Aeris_225') || strcmp(folderName, 'Aeris_259')
                    lat_col = 'Aeris_lat_N';
                    lon_col = 'Aeris_long_W';
                elseif strcmp(folderName, 'MGGA')
                    lat_col = 'MGGA_lat_N';
                    lon_col = 'MGGA_long_W';
                else
                    fprintf('Skipping: %s (Unknown folder type)\n', folderName);
                    continue;
                end

                % Check if the required columns exist before calculations
                if any(strcmp(data.Properties.VariableNames, lat_col)) && any(strcmp(data.Properties.VariableNames, lon_col))

                    % Calculate initial bearing for each row
                    data.bearing = arrayfun(@(lat1, lon1, lat2, lon2) calculate_initial_bearing(lat1, lon1, lat2, lon2), ...
                                            data.Lat, data.Lon, data.(lat_col), data.(lon_col));

                    % Write the updated table back to the same CSV file
                    writetable(data, filename);
                    fprintf('Processed: %s\n', filename);
                else
                    fprintf('Skipping: %s (Missing coordinate columns)\n', filename);
                end
            else
                fprintf('Skipping: %s (Lat/Lon columns missing)\n', filename);
            end
        end
    end
end



%% Step 4: Step 3a: Rotate 3D wind vectors: bLs uses L-calculated by first rotating vectors
clear all;
close all;

% Define the parent directory containing the folders
parentPath = '\\csunas01.colostate.edu\research\METEC\User\mbua\ADED\OTM33_Gaussian\Event Tables';

% Specify the main folders to process
folders = {'MGGA'};

% Specify the subfolders (time intervals)
time_intervals = { '30 minutes'};

% Loop through each folder
for f = 1:numel(folders)
    folderName = folders{f};
    
    % Loop through each time interval subfolder
    for t = 1:numel(time_intervals)
        timeFolder = time_intervals{t};
        
        % Construct the full path to the time interval folder
        path = fullfile(parentPath, folderName, 'Emission Sources', 'Single_Point', timeFolder);
        
        % Check if the folder exists before proceeding
        if ~isfolder(path)
            fprintf('Skipping: %s (Folder not found)\n', path);
            continue;
        end

        % Get a list of all CSV files in the directory
        files = dir(fullfile(path, '*.csv'));

        % Loop over each file
        for k = 1:numel(files)
            filename = fullfile(path, files(k).name);

            % Load data from the current file
            data = readtable(filename); % Load the data into a table

            % Ensure required columns exist before performing transformations
            if all(ismember({'U', 'V', 'W'}, data.Properties.VariableNames))
                
                % Perform transformations: 1st rotation
                avg_u = mean(data.U, 'omitnan'); % Average U (ms⁻¹)
                avg_v = mean(data.V, 'omitnan'); % Average V (ms⁻¹)

                theta1 = atan(avg_u / avg_v); % 1st transformation angle

                data.V1 = cos(theta1) .* data.V + sin(theta1) .* data.U;
                data.U1 = -sin(theta1) .* data.V + cos(theta1) .* data.U;

                % 2nd rotation
                avg_w = mean(data.W, 'omitnan'); % Average W (ms⁻¹)
                avg_v1 = mean(data.V1, 'omitnan'); % Average V1 (ms⁻¹)

                theta2 = atan(avg_w / avg_v1); % 2nd transformation angle

                data.V2 = cos(theta2) .* data.V1 + sin(theta2) .* data.W;
                data.W1 = -sin(theta2) .* data.V1 + cos(theta2) .* data.W;

                % Calculate rotated wind speed
                data.WS_rotated = sqrt(data.V2.^2 + data.U1.^2);

                % Save the transformed data back to the same file
                writetable(data, filename);
                fprintf('Processed: %s\n', filename);
            else
                fprintf('Skipping: %s (Missing required columns U, V, or W)\n', filename);
            end
        end
    end
end


%% Step 5: Calculate Monin-Obukhov length (L)
clear all;
close all;

% Define the parent directory containing the folders
parentPath = '\\csunas01.colostate.edu\research\METEC\User\mbua\ADED\OTM33_Gaussian\Event Tables';

% Specify the main folders to process
folders = {'MGGA'}; 

% Specify the subfolders (time intervals)
time_intervals = {'30 minutes'};

% Constants
g = 9.81;      % Acceleration due to gravity (m/s^2)
kv = 0.41;     % von Kármán constant

% Loop through each folder
for f = 1:numel(folders)
    folderName = folders{f};
    
    % Loop through each time interval subfolder
    for t = 1:numel(time_intervals)
        timeFolder = time_intervals{t};
        
        % Construct the full path to the time interval folder
        path = fullfile(parentPath, folderName, 'Emission Sources', 'Single_Point', timeFolder);
        
        % Check if the folder exists before proceeding
        if ~isfolder(path)
            fprintf('Skipping: %s (Folder not found)\n', path);
            continue;
        end

        % Get a list of all CSV files in the directory
        files = dir(fullfile(path, '*.csv'));

        % Loop over each file
        for k = 1:numel(files)
            filename = fullfile(path, files(k).name);

            % Read data from the current file
            currentTable = readtable(filename);

            % Ensure required columns exist before calculations
            requiredColumns = {'AT', 'BP', 'U1', 'V2', 'W1'};
            if ~all(ismember(requiredColumns, currentTable.Properties.VariableNames))
                fprintf('Skipping: %s (Missing required columns)\n', filename);
                continue;
            end
            
            % Part 1: Calculate surface turbulent flux of sensible heat 
            
            % Calculate potential temperature (Tp)
            Tp = currentTable.AT .* (1000 ./ currentTable.BP) .^ 0.286;

            % Ensure no NaN values affect calculations
            w1_mean = mean(currentTable.W1, 'omitnan');
            Tp_mean = mean(Tp, 'omitnan');

            % Calculate surface turbulent flux of sensible heat (w'Tp')
            wTp = sum((Tp - Tp_mean) .* (currentTable.W1 - w1_mean), 'omitnan') / (height(currentTable) - 1);

            % Add wTp column to table
            currentTable.wTp = repmat(wTp, height(currentTable), 1);

            % Part 2: Calculate surface friction velocity (u*)
            
            u1_mean = mean(currentTable.U1, 'omitnan');
            v2_mean = mean(currentTable.V2, 'omitnan');

            % Compute covariance terms
            cov_uw = sum((currentTable.U1 - u1_mean) .* (currentTable.W1 - w1_mean), 'omitnan') / (height(currentTable) - 1);
            cov_vw = sum((currentTable.V2 - v2_mean) .* (currentTable.W1 - w1_mean), 'omitnan') / (height(currentTable) - 1);

            % Calculate friction velocity (u*)
            uast_value = sqrt(cov_uw^2 + cov_vw^2)^(1/4);

            % Add Uast column to table
            currentTable.Uast = repmat(uast_value, height(currentTable), 1);

            % Ensure no division by zero in Monin-Obukhov length calculation
            if wTp == 0
                currentTable.L = NaN(height(currentTable), 1);
            else
                % Calculate Monin-Obukhov length (L)
                currentTable.L = repmat(-((uast_value.^3 * Tp_mean) ./ (kv * g * wTp)),height(currentTable),1);
            end

            % Save the updated table back to the file
            writetable(currentTable, filename);
            fprintf('Processed: %s\n', filename);
        end
    end
end



%% Step 9: Write the entire table if any difference is within angle thresholds (circular averaging for WD)
clear all;
close all;

% Define the parent directory containing the folders
parentPath = '\\csunas01.colostate.edu\research\METEC\User\mbua\ADED\OTM33_Gaussian\Event Tables';

% Specify the folders to process
folders = {'MGGA'};

% Define the angle thresholds
angleThresholds = [5 10 20];

% Specify the time interval subfolders
time_intervals = {'5 minutes','15 minutes', '30 minutes'};

% Loop through each folder (e.g., MGGA)
for f = 1:numel(folders)
    folderName = folders{f};

    % Loop through each time interval subfolder
    for t = 1:numel(time_intervals)
        timeFolder = time_intervals{t};

        % Define the base file path for the current time interval
        basePath = fullfile(parentPath, folderName, 'Emission Sources', 'Single_Point', timeFolder);

        % Check if the folder exists before proceeding
        if ~isfolder(basePath)
            fprintf('Skipping: %s (Folder not found)\n', basePath);
            continue;
        end

        % Get a list of all CSV files in the directory
        fileList = dir(fullfile(basePath, '*.csv'));

        % Loop through each angle threshold
        for k = 1:numel(angleThresholds)
            angle = angleThresholds(k);

            % Create a new directory for quantified files if it doesn't exist
            quantifiedPath = fullfile(basePath, [num2str(angle) ' degrees']);
            if ~exist(quantifiedPath, 'dir')
                mkdir(quantifiedPath);
            end

            % Loop through each CSV file
            for i = 1:numel(fileList)
                filename = fullfile(basePath, fileList(i).name);

                % Read data from the current file
                data = readtable(filename);

                % Ensure required columns exist before calculations
                requiredColumns = {'WD', 'bearing'};
                if ~all(ismember(requiredColumns, data.Properties.VariableNames))
                    fprintf('Skipping: %s (Missing required columns: WD or bearing)\n', filename);
                    continue;
                end

                % Circular averaging of wind direction (WD)
                sinSum = sum(sind(data.WD), 'omitnan'); % Sum of sines
                cosSum = sum(cosd(data.WD), 'omitnan'); % Sum of cosines
                avgWD = atan2d(sinSum, cosSum); % Circular mean in degrees
                if avgWD < 0
                    avgWD = avgWD + 360; % Ensure avgWD is in [0, 360)
                end

                % Calculate the difference between data.downwind and avgWD
                data.avg_WD = repmat(avgWD, height(data), 1);
                data.downwind = mod(data.bearing - 180, 360);
                data.diff = abs(data.downwind - avgWD);
                data.diff = min(data.diff, 360 - data.diff); % Handle angular wrap-around

                % Check if any difference is within the threshold
                if any(data.diff <= angle)
                    % Write the entire table to a new CSV file in the quantifiedPath
                    [~, fileNameWithoutExtension, extension] = fileparts(filename);
                    newFilename = fullfile(quantifiedPath, [fileNameWithoutExtension extension]);
                    writetable(data, newFilename);
                    fprintf('Processed: %s (Angle Threshold: %d degrees)\n', newFilename, angle);
                end
            end
        end
    end
end
